# Librecast - Distributed Applications with IPv6 Multicast

![Librecast Logo](https://www.librecast.net/media/librecast.hex.svg)

<a href="https://scan.coverity.com/projects/librecast">
  <img alt="Coverity Scan Build Status" src="https://scan.coverity.com/projects/25262/badge.svg"/>
</a>

<a href="https://repology.org/project/librecast/versions">
  <img src="https://repology.org/badge/vertical-allrepos/librecast.svg" alt="Packaging status" align="right">
</a>

# README

Librecast is a C multicast library which aims to make working with multicast easier.

Librecast  extends  IPv6  multicast to provide a multicast communication layer with support for encodings, encryption, file syncing, router topologies and overlay multicast.

<a id="contents"></a>
## Contents

- [Background](#background)
- [License](#license)
- [Obtaining Librecast](#obtain)
- [Installing Librecast](#install)
- [Getting Started - Writing Librecast Programs](#getting-started)
- [Bugs (also Questions and Feature Requests)](#bugs)
- [Contact Us](#contact)
	- [Email](#contact-email)
	- [Fediverse (Mastodon)](#contact-fediverse)
	- [IRC](#contact-irc)
	- [Matrix](#contact-matrix)
- [Code of Conduct](#coc)
- [Contributing](#contributing)

<a id="background"></a>
## Background

The [Librecast Project](https://librecast.net/) aims to enable universal group communication, and to provide the libraries and tools we need for that.

Universal group communication is a requirement for human rights to flourish. Problems, such as climate change are global and affect the rights of all humans. Our ability to communicate, regardless of frontiers, is essential to our need to be informed, to organize, to promote and enjoy our rights, and to hold accountable those that would violate our rights.

All organizations and power structures must be held accountable for human rights to have any meaning. To monitor and hold accountable global power structures such as corporations, governments and NGOs, we require global communication that is not subject to interference, interception or control by those powers.

Our Internet is a requirement, not an option, for our global civilization, but that Internet is not fit for purpose. We need a Next Generation Internet (NGI) built around group communication and the tools to use it effectively.

The Internet we have today is built for one-to-one communication (unicast), but our communication needs are many-to-many (multicast). Group communication built on top of unicast is inefficient and cumbersome, and often relies on centralized 3rd party servers controlled by large corporations.

Together, we can change that and build a rights-respecting Next Generation Internet for all humans.

<a id="license"></a>
## License

This work is dual-licensed under GPL 2.0 and GPL 3.0.

`SPDX-License-Identifier: GPL-2.0-only OR GPL-3.0-only`

<a id="obtain"></a>
## Obtaining Librecast

Thanks to `vagrantc`, many of Librecast's libraries and tools are packaged for Debian, GUIX and Debian derivatives such as Ubuntu.

Librecast is also available in [NixOS](https://search.nixos.org/packages?channel=24.11&from=0&size=50&sort=relevance&type=packages&query=librecast) thanks to the team at Summer of Nix.

You can obtain the source code from Codeberg and SourceHut:

### liblibrecast

- [Codeberg](https://codeberg.org/librecast/)
- [SourceHut](https://sr.ht/~librecast/librecast/sources)

## Installing Librecast

See [INSTALL.md](INSTALL.md).

<a id="getting-started"></a>
## Getting Started - Programming with Librecast

Let's look at two simple Librecast programs in C for sending and receiving data
over IPv6 multicast:

### multicast sending program
```
/* Librecast sender example */

#include <librecast.h>

int main(void)
{
        char data[] = "I have a dream";
        ssize_t bytes;
        int rc = EXIT_FAILURE;  /* pessimistic, aren't we? */

        /* we start with a Librecast Context (lc_ctx_t) */
        lc_ctx_t *lctx = lc_ctx_new();
        if (!lctx) return rc;

        /* create a Librecast Socket (IPv6) */
        lc_socket_t *sock = lc_socket_new(lctx);
        if (!sock) goto free_ctx;

        /* create a Librecast Channel */
        lc_channel_t *chan = lc_channel_new(lctx, "my very first Librecast channel");

        /* bind the Channel to the Socket */
        lc_channel_bind(sock, chan);

        /* enable loopback, so we receive our own packets on the same host */
        lc_socket_loop(sock, 1);

        /* (optional) enable RaptorQ encoding (RFC6330) */
        lc_channel_coding_set(chan, LC_CODE_FEC_RQ | LC_CODE_FEC_OTI);
        lc_channel_rq_overhead(chan, RQ_OVERHEAD + 5);

        /* (optional) rate-limit sending */
        lc_channel_ratelimit(chan, 104857600 /* 100Mbps */ , 0);

        /* send some multicast data */
        bytes = lc_channel_send(chan, data, sizeof data, 0);
        if (bytes == -1) {
                perror("lc_channel_send");
                goto free_ctx;
        }
        printf("sent: '%s' (%zi bytes)\n", data, bytes);

        rc = EXIT_SUCCESS;
free_ctx:
        /* free the Context. This also frees all Sockets, Channels, Routers etc.
         * created with that Context */
        lc_ctx_free(lctx);
        return rc;
}
```

### multicast receiver program

```
/* SPDX-License-Identifier: GPL-2.0-only OR GPL-3.0-only */
/* Copyright (c) 2025 Brett Sheffield <bacs@librecast.net> */

/* Librecast receiver example */

#include <librecast.h>

int main(void)
{
        char data[BUFSIZ] = {0};
        ssize_t bytes;
        int rc = EXIT_FAILURE;  /* pessimistic, aren't we? */

        /* we start with a Librecast Context (lc_ctx_t) */
        lc_ctx_t *lctx = lc_ctx_new();
        if (!lctx) return rc;

        /* create a Librecast Socket (IPv6) */
        lc_socket_t *sock = lc_socket_new(lctx);
        if (!sock) goto free_ctx;

        /* create a Librecast Channel */
        lc_channel_t *chan = lc_channel_new(lctx, "my very first Librecast channel");

        /* bind the Channel to the Socket */
        lc_channel_bind(sock, chan);

        /* join the Channel or we won't receive any data */
        lc_channel_join(chan);

        /* (optional) enable RaptorQ encoding (RFC6330) */
        lc_channel_coding_set(chan, LC_CODE_FEC_RQ | LC_CODE_FEC_OTI);

        /* recv some multicast data */
        bytes = lc_channel_recv(chan, data, sizeof data, 0);
        if (bytes == -1) goto free_ctx;
        printf("received: '%s' (%zi bytes)\n", data, bytes);

        rc = EXIT_SUCCESS;
free_ctx:
        /* free the Context. This also frees all Sockets, Channels, Routers etc.
         * created with that Context */
        lc_ctx_free(lctx);
        return rc;
}
```

These programs can be found in the `examples/` directory of the Librecast source
code.  There is a lot more example code which exercises the various capabilities
of Librecast in the `test/` directory.

The source code for [lcagent](https://codeberg.org/librecast/lcagent) and
[lcsync](https://codeberg.org/librecast/lcsync) also demonstrate some of what a
Librecast program can do.

We are planning a Librecast Programming Guide, but in the meantime, read the
headers and man pages, look at the tests and examples, and if you get stuck,
feel free to [ask](#contact).

<a id="contact"></a>
## Contact Us

One of the great things about Free/Open Source Software is that you can talk with the people who make it.  A project like this is about community as well as code, so we want to make you feel welcome. That's why we have a [Code of Conduct](CODE_OF_CONDUCT.md) and [Contributing](CONTRIBUTING.md) guidelines.  Please do take a moment to read them.

There are several ways to get in touch with us. If you think you've found a bug, please see the [Bugs](#Bugs) section for how to report it. You can also use our bug tracker to suggest a feature, or to ask a question, especially if you think the answer to that question may be of interest to others.

For anything else, pick one of the contact methods below.

<a id="contact-email"></a>
### Email

We have low-traffic mailing lists for
[announcements](https://lists.sr.ht/~librecast/librecast-announce), general
[discussion](https://lists.sr.ht/~librecast/librecast-discuss) related to the
Librecast Project and
[development](https://lists.sr.ht/~librecast/librecast-devel).

In addition, you can contact:

- [bugs@librecast.net](mailto:bugs@librecast.net) to report a bug
- [coc@librecast.net](mailto:coc@librecast.net) for any questions related to [Code of Conduct](CODE_OF_CONDUCT.md) or to make a report
- [security@librecast.net](mailto:security@librecast.net) for security reports

<a id="contact-fediverse"></a>
### Fediverse (Mastodon)

Follow us on the Fediverse. We announce new releases and project updates on Mastodon. You can find us here: [@librecast@chaos.social](https://chaos.social/@librecast).

A huge thanks to Leah and rixx for hosting our project on chaos.social and for
providing a stable and well moderated home for our community.

<a id="contact-irc"></a>
### IRC channel

`#librecast` on irc.libera.chat

If you have a question, please be patient. An answer might take a few hours
depending on time zones and whether anyone on the team is available at that
moment.  We're always connected, so we'll notice eventually.

Thanks to the <a href="https://libera.chat/">Libera.Chat</a> team for hosting our IRC channels.

<a id="contact-matrix"></a>
### Matrix

We have a [Librecast Matrix Room](https://matrix.to/#/#librecast:matrix.org).

<a id="coc"></a>
## Code of Conduct

See [Code of Conduct](CODE_OF_CONDUCT.md).

<a id="contributing"></a>
## Contributing

See [Contributing](CONTRIBUTING.md).

<a id="bugs"></a>
## Bugs (also Questions and Feature Requests)

New issues can be raised at:

https://bugs.librecast.net/librecast

It's okay to raise an issue to ask a question.  You can also email or ask on
IRC. See [Contact Us](#contact-us).

<hr />

<p class="bigbreak">
This project was funded through the <a href="https://nlnet.nl/discovery">NGI Zero Discovery</a>, <a href="https://nlnet.nl/assure">NGI Assure</a> and <a href="https://nlnet.nl/core">NGI Zero Core</a> funds, established by NLnet with financial support from the European
Commission's <a href="https://ngi.eu">Next Generation Internet</a> programme. *See the NLnet open calls page to <a href="https://nlnet.nl/propose/">apply for funding</a> for your project*
</p>

<p>
  <a href="https://nlnet.nl/project/Librecast-OverlayMulticast/">
      <img align="left" width="250" src="https://nlnet.nl/logo/banner.png" alt="Logo NLnet: abstract logo of four people seen from above" />
  </a>
  <a href="https://ngi.eu/">
      <img align="right" width="250" src="https://nlnet.nl/image/logos/NGI0Core_tag.svg" alt="NGI Core Logo" />
  </a>
</p>
